import { prisma } from "../../lib/prisma.js";
export async function createOrUpdateOrder(params) {
    const room = await prisma.room.findUnique({
        where: { id: params.roomId },
        include: { menuItems: true },
    });
    if (!room)
        throw new Error("Room not found");
    if (room.orderDeadline && new Date() > room.orderDeadline)
        throw new Error("Order deadline passed");
    return prisma.$transaction(async (tx) => {
        const order = await tx.order.upsert({
            where: {
                roomId_memberName: {
                    roomId: params.roomId,
                    memberName: params.memberName,
                },
            },
            update: {
                items: {
                    deleteMany: {},
                },
            },
            create: {
                roomId: params.roomId,
                memberName: params.memberName,
            },
        });
        for (const menuItemId of params.menuItemIds) {
            const menu = room.menuItems.find((m) => m.id === menuItemId);
            if (!menu)
                throw new Error("Invalid menu item");
            await tx.orderItem.create({
                data: {
                    orderId: order.id,
                    itemName: menu.name,
                    price: menu.price,
                    qty: 1, // fixed to 1
                },
            });
        }
        return order;
    });
}
export async function deleteOrder(params) {
    const order = await prisma.order.findUnique({
        where: {
            roomId_memberName: {
                roomId: params.roomId,
                memberName: params.memberName,
            },
        },
    });
    if (!order)
        throw new Error("Order not found");
    return prisma.order.delete({
        where: { id: order.id },
    });
}
